// My standard defines for all pulse files.

#include <pulse_os.h>
#include <pulse_types.h>

typedef unsigned char u8;
typedef signed char s8;
typedef uint16_t u16;
typedef int16_t s16;
typedef uint32_t u32;
typedef int32_t s32;
//
// Defines the morse functions.
// Hook up, passing in current millisec
//    morse_buttondown(int ms)
//    morse_buttonup(int ms)
//    morse_mainloop(int ms)
//
// Read with, which return 0 if none on stack.
// u8 morse_getchar()
//
// RAM use: 11 bytes
//

//
// Prototypes
//
u8 morse_getchar();
void morse_mainloop(int ms);
void morse_buttonup(int ms);
void morse_buttondown(int ms);
void morse_reset();

//
// Implementation
//

#define		MORSE_DOT		250
#define		MORSE_ENDOFSYMBOL	400
#define		MORSE_SPACEPUSH		1100

u32 morse_downtime = 0;
u32 morse_uptime = 0;

u8	morse_seq;
u8	morse_symbols = 0;
u8	morse_char = 0;

void
morse_pushdot()
{
    // zero by default
    morse_symbols++;
}

void
morse_pushdash()
{
    morse_seq |= 1 << morse_symbols;
    morse_symbols++;
}

void
morse_pushchar(u8 c)
{
    morse_char = c;
}

void
morse_pushcode()
{
    if (morse_symbols == 0)
	return;

    if (morse_symbols < 5)
    {
	const char *lut =
	    //
	       " "
	    //  oI
	       "et"
	    //  oIoI
	    //  ooII
	       "inam"
	    //  oIoIoIoI
	    //  ooIIooII
	    //  ooooIIII
	       "sdrgukwo"
	    //  oIoIoIoIoIoIoIoI
	    //  ooIIooIIooIIooII
	    //  ooooIIIIooooIIII
	    //  ooooooooIIIIIIII
	       "hblzfcp-vx-q-yj-"
	    ;
	int	off;
	off = 1 << morse_symbols;
	off--;
	off += morse_seq;
	morse_pushchar(lut[off]);
    }

    // Longer lengths take special codes.
    else if (morse_symbols == 5)
    {
	// NOTE: These are all backwards bit patterns!
	switch (morse_seq)
	{
	    // Number patterns
	    case 0b11110:
		morse_pushchar('1');
		break;
	    case 0b11100:
		morse_pushchar('2');
		break;
	    case 0b11000:
		morse_pushchar('3');
		break;
	    case 0b10000:
		morse_pushchar('4');
		break;
	    case 0b00000:
		morse_pushchar('5');
		break;
	    case 0b00001:
		morse_pushchar('6');
		break;
	    case 0b00011:
		morse_pushchar('7');
		break;
	    case 0b00111:
		morse_pushchar('8');
		break;
	    case 0b01111:
		morse_pushchar('9');
		break;
	    case 0b11111:
		morse_pushchar('0');
		break;

	    // Punctuation
	    case 0b01001:
		morse_pushchar('/');
		break;
	    case 0b01101:
		morse_pushchar('(');
		break;
	    case 0b00010:
		morse_pushchar('&');
		break;
	    case 0b10001:
		morse_pushchar('=');
		break;
	    case 0b01010:
		morse_pushchar('+');
		break;
	}
    }
    else if (morse_symbols == 6)
    {
	// NOTE: These are all backwards bit patterns!
	switch (morse_seq)
	{
	    case 0b000000:
		// All e's, error code.
		morse_pushchar('\b');
		break;
	    case 0b101010:
		morse_pushchar('.');
		break;
	    case 0b110011:
		morse_pushchar(',');
		break;
	    case 0b001100:
		morse_pushchar('?');
		break;
	    case 0b011110:
		morse_pushchar('\'');
		break;
	    case 0b101011:
		morse_pushchar('!');
		break;
	    case 0b101101:
		morse_pushchar(')');
		break;
	    case 0b000111:
		morse_pushchar(':');
		break;
	    case 0b010101:
		morse_pushchar(';');
		break;
	    case 0b100001:
		morse_pushchar('-');
		break;
	    case 0b101100:
		morse_pushchar('_');
		break;
	    case 0b010010:
		morse_pushchar('"');
		break;

	    // Recent addition from 2004.
	    case 0b010110:
		morse_pushchar('@');
		break;

	}
    }
    else if (morse_symbols == 7)
    {
	switch (morse_seq)
	{
	    case 0b0000000:
		// All e's, error code.
		morse_pushchar('\b');
		break;
	    case 0b1001000:
		morse_pushchar('$');
		break;
	}
    }
    // 6 or more e's is considered error, ie, backspace, so
    // we detect
    else if (!morse_seq)
    {
	morse_pushchar('\b');
    }

    // Reset in any case.
    morse_symbols = 0;
    morse_seq = 0;
}

void 
morse_process(bool signal, int lenms)
{
    if (signal)
    {
	if (lenms < MORSE_DOT)
	    morse_pushdot();
	else
	    morse_pushdash();
    }
    else
    {
	if (lenms > MORSE_ENDOFSYMBOL)
	{
	    morse_pushcode();
	}
	if (lenms > MORSE_SPACEPUSH)
	{
	    morse_uptime = 0;
	    morse_pushchar(' ');
	}
    }
}

void
morse_buttondown(int ms)
{
    morse_uptime = 0;
    morse_downtime = ms;
}

void
morse_buttonup(int ms)
{
    morse_uptime = ms;
    morse_process(true, morse_uptime - morse_downtime);
    morse_downtime = 0;
}

void
morse_mainloop(int ms)
{
    if (morse_uptime)
    {
	morse_process(false, ms - morse_uptime);
    }
}

u8
morse_getchar()
{
    u8		c;
    c = morse_char;
    morse_char = 0;
    return c;
}

void
morse_reset()
{
    morse_char = 0;
    morse_seq = 0;
    morse_uptime = 0;
}
//
// Defines the rand functions.
// RAM use: 4 bytes
//

//
// Prototypes
//
void rand_seed();
int rand_choice(int nchoice);
void rand_angletodir(int angle, int *dx, int *dy);
void rand_dir(int *dx, int *dy);

//
// implementation
//
int rand_curseed = 132;

#define RAND() (rand_curseed = (rand_curseed * 1013904223 + 1664525))

void
rand_seed()
{
    struct pulse_time_tm current_time;
    pulse_get_time_date(&current_time);
    rand_curseed = (current_time.tm_hour * 60 + current_time.tm_min) * 60 + current_time.tm_sec;
}

int
rand_choice(int nchoice)
{
    if (nchoice < 0)
	return -rand_choice(-nchoice);
    if (nchoice < 2)
	return 0;

    u32		val = RAND();

    // eat bad bits and avoid sign
    val >>= 4;

    return val % nchoice;
}

void
rand_angletodir(int angle, int *dx, int *dy)
{
    if (angle & 2)
	*dx = *dy = -1;
    else
	*dx = *dy = 1;

    if (angle & 1)
	*dx = 0;
    else
	*dy = 0;
}

#define FORALL_4DIR(dx, dy) \
    for (int lcl_angle = 0; rand_angletodir(lcl_angle, &dx, &dy), lcl_angle < 4; lcl_angle++)

void
rand_dir(int *dx, int *dy)
{
    u32		val = RAND();

    // Mix in the full entropy.
    val ^= val >> 16;
    val ^= val >> 8;
    val ^= val >> 4;
    val ^= val >> 2;

    rand_angletodir(val, dx, dy);
}

const unsigned char alphabet_Tiles[960] = {
0x00,0x00,0x78,0x0c,0x7c,0xcc,0x76,0x00,0xe0,0x60,0x7c,0x66,0x66,0x66,0xbc,0x00,
0x00,0x00,0x78,0xcc,0xc0,0xcc,0x78,0x00,0x1c,0x0c,0x0c,0x7c,0xcc,0xcc,0x76,0x00,
0x00,0x00,0x78,0xcc,0xfc,0xc0,0x78,0x00,0x38,0x6c,0x60,0xf8,0x60,0x60,0xf8,0x00,
0x00,0x00,0x76,0xcc,0x7c,0x0c,0xf8,0x00,0xe0,0x60,0x6c,0x76,0x66,0x66,0xe6,0x00,
0x18,0x00,0x78,0x18,0x18,0x18,0x7e,0x00,0x0c,0x00,0x3c,0x0c,0x0c,0x0c,0x78,0x00,
0xe0,0x60,0x66,0x6c,0x78,0x6c,0xe6,0x00,0x78,0x18,0x18,0x18,0x18,0x18,0x7e,0x00,
0x00,0x00,0xec,0xfe,0xd6,0xc6,0xc6,0x00,0x00,0x00,0xf8,0xcc,0xcc,0xcc,0xcc,0x00,
0x00,0x00,0x78,0xcc,0xcc,0xcc,0x78,0x00,0x00,0x00,0xdc,0x66,0x7c,0x60,0xf0,0x00,
0x00,0x00,0x76,0xcc,0x7c,0x0c,0x1e,0x00,0x00,0x00,0xd8,0x6c,0x6c,0x60,0xf0,0x00,
0x00,0x00,0x7c,0xc0,0x78,0x0c,0xf8,0x00,0x10,0x30,0x7c,0x30,0x30,0x34,0x18,0x00,
0x00,0x00,0xcc,0xcc,0xcc,0xcc,0x76,0x00,0x00,0x00,0xcc,0xcc,0xcc,0x78,0x30,0x00,
0x00,0x00,0xc6,0xc6,0xd6,0xfe,0x6c,0x00,0x00,0x00,0xc6,0x6c,0x38,0x6c,0xc6,0x00,
0x00,0x00,0xcc,0xcc,0x7c,0x0c,0xf8,0x00,0x00,0x00,0xfc,0x98,0x30,0x64,0xfc,0x00,
0x00,0x6c,0xfe,0xfe,0x7c,0x38,0x10,0x00,0x10,0x10,0x38,0x38,0x7c,0xfe,0x7c,0x00,
0x7c,0x04,0x14,0x34,0x7c,0x30,0x10,0x00,0x42,0x7e,0x42,0x42,0x42,0x7e,0x42,0x00,
0x30,0x78,0xcc,0xcc,0xfc,0xcc,0xcc,0x00,0xfc,0x66,0x66,0x7c,0x66,0x66,0xfc,0x00,
0x3c,0x66,0xc0,0xc0,0xc0,0x66,0x3c,0x00,0xfc,0x6c,0x66,0x66,0x66,0x6c,0xfc,0x00,
0xfe,0x62,0x68,0x78,0x68,0x62,0xfe,0x00,0xfe,0x62,0x68,0x78,0x68,0x60,0xf0,0x00,
0x3c,0x66,0xc0,0xc0,0xce,0x66,0x3e,0x00,0xcc,0xcc,0xcc,0xfc,0xcc,0xcc,0xcc,0x00,
0xfc,0x30,0x30,0x30,0x30,0x30,0xfc,0x00,0x1e,0x0c,0x0c,0x0c,0xcc,0xcc,0x78,0x00,
0xe6,0x66,0x6c,0x78,0x6c,0x66,0xe6,0x00,0xf0,0x60,0x60,0x60,0x62,0x66,0xfe,0x00,
0xc6,0xee,0xfe,0xd6,0xc6,0xc6,0xc6,0x00,0xc6,0xe6,0xf6,0xde,0xce,0xc6,0xc6,0x00,
0x38,0x6c,0xc6,0xc6,0xc6,0x6c,0x38,0x00,0xfc,0x66,0x66,0x7c,0x60,0x60,0xf0,0x00,
0x78,0xcc,0xcc,0xcc,0xdc,0x78,0x1c,0x00,0xfc,0x66,0x66,0x7c,0x78,0x6c,0xe6,0x00,
0x78,0xcc,0xe0,0x38,0x1c,0xcc,0x78,0x00,0xfc,0xb4,0x30,0x30,0x30,0x30,0x78,0x00,
0xcc,0xcc,0xcc,0xcc,0xcc,0xcc,0xfc,0x00,0xcc,0xcc,0xcc,0xcc,0xcc,0x78,0x30,0x00,
0xc6,0xc6,0xc6,0xd6,0xfe,0xee,0xc6,0x00,0xc6,0xc6,0x6c,0x38,0x6c,0xc6,0xc6,0x00,
0xcc,0xcc,0xcc,0x78,0x30,0x30,0x78,0x00,0xfe,0xcc,0x98,0x30,0x62,0xc6,0xfe,0x00,
0x00,0x10,0x30,0x7c,0x30,0x10,0x00,0x00,0x00,0x10,0x18,0x7c,0x18,0x10,0x00,0x00,
0x00,0x10,0x38,0x7c,0x10,0x10,0x00,0x00,0x00,0x10,0x10,0x7c,0x38,0x10,0x00,0x00,
0x78,0xcc,0xdc,0xfc,0xec,0xcc,0x78,0x00,0x30,0xf0,0x30,0x30,0x30,0x30,0xfc,0x00,
0x78,0xcc,0x0c,0x38,0x60,0xcc,0xfc,0x00,0x78,0xcc,0x0c,0x38,0x0c,0xcc,0x78,0x00,
0x1c,0x3c,0x6c,0xcc,0xfe,0x0c,0x0c,0x00,0xfc,0xc0,0xf8,0x0c,0x0c,0xcc,0x78,0x00,
0x38,0x60,0xc0,0xf8,0xcc,0xcc,0x78,0x00,0xfc,0xcc,0x0c,0x18,0x30,0x60,0x60,0x00,
0x78,0xcc,0xcc,0x78,0xcc,0xcc,0x78,0x00,0x78,0xcc,0xcc,0x7c,0x0c,0x18,0x70,0x00,
0x30,0x78,0x78,0x30,0x30,0x00,0x30,0x00,0x7c,0xc6,0xde,0xde,0xde,0xc0,0x78,0x00,
0x6c,0x6c,0xfe,0x6c,0xfe,0x6c,0x6c,0x00,0x30,0x7c,0xc0,0x78,0x0c,0xf8,0x30,0x00,
0x00,0xc6,0xcc,0x18,0x30,0x66,0xc6,0x00,0x10,0x38,0x6c,0xc6,0x00,0x00,0x00,0x00,
0x38,0x6c,0x38,0x76,0xdc,0xcc,0x76,0x00,0x00,0x24,0x18,0x7e,0x18,0x24,0x00,0x00,
0x18,0x30,0x60,0x60,0x60,0x30,0x18,0x00,0x60,0x30,0x18,0x18,0x18,0x30,0x60,0x00,
0x00,0x00,0xfc,0x00,0xfc,0x00,0x00,0x00,0x00,0x00,0x00,0xfc,0x00,0x00,0x00,0x00,
0xc0,0x60,0x30,0x18,0x0c,0x06,0x02,0x00,0x30,0x30,0x60,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x30,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,
0x06,0x0c,0x18,0x30,0x60,0xc0,0x80,0x00,0x00,0x30,0x30,0xfc,0x30,0x30,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0xfe,0x00,0x18,0x18,0x18,0x00,0x18,0x18,0x18,0x00,
0x6c,0x6c,0x6c,0x00,0x00,0x00,0x00,0x00,0x0c,0x18,0x30,0x60,0x30,0x18,0x0c,0x00,
0x60,0x30,0x18,0x0c,0x18,0x30,0x60,0x00,0x78,0xcc,0x0c,0x18,0x30,0x00,0x30,0x00,
0x78,0x60,0x60,0x60,0x60,0x60,0x78,0x00,0x78,0x18,0x18,0x18,0x18,0x18,0x78,0x00,
0x1c,0x30,0x30,0xe0,0x30,0x30,0x1c,0x00,0xe0,0x30,0x30,0x1c,0x30,0x30,0xe0,0x00,
0x30,0x30,0x18,0x00,0x00,0x00,0x00,0x00,0x76,0xdc,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x30,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x30,0x00,
0x0c,0x18,0x30,0x60,0x30,0x18,0x0c,0x00,0x60,0x30,0x18,0x0c,0x18,0x30,0x60,0x00,
0x00,0x00,0x30,0x30,0x00,0x30,0x60,0x00,0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x00,
0x84,0xfc,0xfc,0xfc,0xfc,0xfc,0x78,0x00,0x00,0x3c,0x3c,0x18,0x3c,0x7e,0x5a,0x00,
0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x20,0x70,0x20,0x20,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};
const unsigned char gfx_backtable[256] = 
{
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
109, 70, 90, 72, 73, 74, 76, 83, 78, 79, 77, 87, 84, 81, 85, 86, 
60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 105, 104, 91, 80, 92, 93, 
71, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 
45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 94, 82, 95, 75, 88, 
98, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 
15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 96, 89, 97, 99, 1, 
26, 27, 56, 57, 58, 59, 28, 29, 106, 107, 108, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1

};
// Automagically generated by enummaker.exe.
// DO NOT EDIT THIS FILE (Yes, I mean you!)
#ifndef __glbdef_h__
#define __glbdef_h__


// Definitions for MOB
enum MOB_NAMES
{
    MOB_NONE,
    MOB_AVATAR,
    MOB_ORC,
    NUM_MOBS
};

// Macros for MOB
#define FOREACH_MOB(x) \
    for ((x) = (MOB_NAMES) 0; \
         (x) < NUM_MOBS; \
         (x) = (MOB_NAMES) ((int)(x)+1))

struct MOB_DEF
{
    const char *name;
    u8 symbol;
    u8 attr;
    u8 melee_damage;
    u8 melee_chance;
    u8 max_hp;
    u8 depth;
    u8 ai;
};

extern const struct MOB_DEF glb_mobdefs[];

// Definitions for AI
enum AI_NAMES
{
    AI_NONE,
    AI_STRAIGHTLINE,
    NUM_AIS
};

// Macros for AI
#define FOREACH_AI(x) \
    for ((x) = (AI_NAMES) 0; \
         (x) < NUM_AIS; \
         (x) = (AI_NAMES) ((int)(x)+1))

// Definitions for TILE
enum TILE_NAMES
{
    TILE_INVALID,
    TILE_NONE,
    TILE_FLOOR,
    TILE_DOWNSTAIRS,
    NUM_TILES
};

// Macros for TILE
#define FOREACH_TILE(x) \
    for ((x) = (TILE_NAMES) 0; \
         (x) < NUM_TILES; \
         (x) = (TILE_NAMES) ((int)(x)+1))

struct TILE_DEF
{
    u8 symbol;
    u8 attr;
    bool ispassable : 1;
    bool istransparent : 1;
};

extern const struct TILE_DEF glb_tiledefs[];

// Definitions for MAPFLAG
enum MAPFLAG_NAMES
{
    MAPFLAG_NONE = 0,
    MAPFLAG_FOV = 128,
    MAPFLAG_MAPPED = 64,
    MAPFLAG_MOB = 32,
    NUM_MAPFLAGS
};

// Macros for MAPFLAG
#define FOREACH_MAPFLAG(x) \
    for ((x) = (MAPFLAG_NAMES) 0; \
         (x) < NUM_MAPFLAGS; \
         (x) = (MAPFLAG_NAMES) ((int)(x)+1))

// Definitions for ATTR
enum ATTR_NAMES
{
    ATTR_NONE,
    ATTR_HILITE,
    ATTR_GOLD,
    ATTR_YELLOW,
    ATTR_PINK,
    ATTR_PURPLE,
    ATTR_NORMAL,
    ATTR_LIGHTBLACK,
    ATTR_OUTOFFOV,
    ATTR_WHITE,
    ATTR_ORANGE,
    ATTR_LIGHTBROWN,
    ATTR_BROWN,
    ATTR_RED,
    ATTR_DKRED,
    ATTR_GREEN,
    ATTR_DKGREEN,
    ATTR_BLUE,
    ATTR_LIGHTBLUE,
    ATTR_TEAL,
    ATTR_CYAN,
    ATTR_DKCYAN,
    ATTR_GREY,
    NUM_ATTRS
};

// Macros for ATTR
#define FOREACH_ATTR(x) \
    for ((x) = (ATTR_NAMES) 0; \
         (x) < NUM_ATTRS; \
         (x) = (ATTR_NAMES) ((int)(x)+1))

struct ATTR_DEF
{
    u8 bg_r;
    u8 bg_g;
    u8 bg_b;
    u8 fg_r;
    u8 fg_g;
    u8 fg_b;
};

extern const struct ATTR_DEF glb_attrdefs[];

// Definitions for ACTION
enum ACTION_NAMES
{
    ACTION_NONE,
    ACTION_RESTART,
    ACTION_BUMP,
    ACTION_WAIT,
    NUM_ACTIONS
};

// Macros for ACTION
#define FOREACH_ACTION(x) \
    for ((x) = (ACTION_NAMES) 0; \
         (x) < NUM_ACTIONS; \
         (x) = (ACTION_NAMES) ((int)(x)+1))
#endif
// Automagically generated by enummaker.exe.
// DO NOT EDIT THIS FILE (Yes, I mean you!)


// Definitions for MOB
const struct MOB_DEF
glb_mobdefs[NUM_MOBS] =
{
  {
    "error",
    '!',
    ATTR_WHITE,
    1,
    75,
    1,
    0,
    AI_NONE,
  },
  {
    "you",
    '@',
    ATTR_WHITE,
    1,
    75,
    10,
    0,
    AI_NONE,
  },
  {
    "orc",
    'o',
    ATTR_GREEN,
    1,
    75,
    1,
    0,
    AI_NONE,
  },
};

// Definitions for TILE
const struct TILE_DEF
glb_tiledefs[NUM_TILES] =
{
  {
    '#',
    ATTR_BROWN,
    false,
    false,
  },
  {
    '&',
    ATTR_NORMAL,
    true,
    true,
  },
  {
    '.',
    ATTR_NORMAL,
    true,
    true,
  },
  {
    '>',
    ATTR_NORMAL,
    true,
    true,
  },
};

// Definitions for ATTR
const struct ATTR_DEF
glb_attrdefs[NUM_ATTRS] =
{
  {
    0,
    0,
    0,
    255,
    255,
    255,
  },
  {
    255,
    255,
    255,
    0,
    0,
    0,
  },
  {
    0,
    0,
    0,
    255,
    196,
    0,
  },
  {
    0,
    0,
    0,
    255,
    255,
    0,
  },
  {
    0,
    0,
    0,
    255,
    128,
    128,
  },
  {
    0,
    0,
    0,
    255,
    128,
    255,
  },
  {
    0,
    0,
    0,
    192,
    192,
    192,
  },
  {
    0,
    0,
    0,
    96,
    96,
    96,
  },
  {
    0,
    0,
    0,
    96,
    96,
    96,
  },
  {
    0,
    0,
    0,
    255,
    255,
    255,
  },
  {
    0,
    0,
    0,
    255,
    192,
    64,
  },
  {
    0,
    0,
    0,
    255,
    192,
    128,
  },
  {
    0,
    0,
    0,
    192,
    128,
    64,
  },
  {
    0,
    0,
    0,
    255,
    32,
    32,
  },
  {
    0,
    0,
    0,
    196,
    32,
    32,
  },
  {
    0,
    0,
    0,
    0,
    255,
    0,
  },
  {
    0,
    0,
    0,
    0,
    196,
    0,
  },
  {
    0,
    0,
    0,
    64,
    64,
    255,
  },
  {
    0,
    0,
    0,
    128,
    128,
    255,
  },
  {
    0,
    0,
    0,
    64,
    128,
    128,
  },
  {
    0,
    0,
    0,
    64,
    255,
    255,
  },
  {
    0,
    0,
    0,
    64,
    192,
    192,
  },
  {
    0,
    0,
    0,
    196,
    196,
    196,
  },
};
//
// gfxengine
//
// RAM use: 10 bytes

s8 	gfx_cursorx = 0;
s8 	gfx_cursory = 0;
color24_t	gfx_bg;
color24_t	gfx_fg;

#define GFX_TEXTWIDTH	12
#define GFX_TEXTHEIGHT	16

// These are special symbols in our character set.  The SYMBOLSTRING
// version is "".
#define SYMBOL_HEART		'\200'
#define SYMBOLSTRING_HEART	"\200"
#define SYMBOL_MAGIC		'\201'
#define SYMBOLSTRING_MAGIC	"\201"
#define SYMBOL_LEFT		'\202'
#define SYMBOLSTRING_LEFT	"\202"
#define SYMBOL_RIGHT		'\203'
#define SYMBOLSTRING_RIGHT	"\203"
#define SYMBOL_UP		'\204'
#define SYMBOLSTRING_UP		"\204"
#define SYMBOL_DOWN		'\205'
#define SYMBOLSTRING_DOWN	"\205"
#define SYMBOL_NEXT		'\206'
#define SYMBOLSTRING_NEXT	"\206"
#define SYMBOL_DLEVEL		'\207'
#define SYMBOLSTRING_DLEVEL	"\207"
#define SYMBOL_AC		'\210'
#define SYMBOLSTRING_AC		"\210"
#define SYMBOL_TRIDUDE		'\211'
#define SYMBOLSTRING_TRIDUDE	"\211"
#define SYMBOL_CURSOR		'\212'
#define SYMBOLSTRING_CURSOR	"\212"
#define SYMBOL_UNIQUE		'\213'
#define SYMBOLSTRING_UNIQUE	"\213"

int
gfx_getchartile(u8 c)
{
    return gfx_backtable[c];
}

void
gfx_reset()
{
    gfx_cursorx = gfx_cursory = 0;
    gfx_bg.red = gfx_bg.green = gfx_bg.blue = 0;
    gfx_fg.red = gfx_fg.green = gfx_fg.blue = 255;
}

void
gfx_putcharCD(int x, int y, u8 c, color24_t fg, color24_t bg)
{
    // Clip.
    if (x < 0 || x >= GFX_TEXTWIDTH)
	return;
    if (y < 0 || y >= GFX_TEXTHEIGHT)
	return;

    int		offset = gfx_getchartile(c);
    offset *= 8;
    u8		data;

    pulse_set_draw_window(x * 8, y * 8, x * 8 + 7, y * 8 + 7);
    for (int sy = 0; sy < 8; sy++)
    {
	data = alphabet_Tiles[offset + sy];
	for (int sx = 0; sx < 8; sx++)
	{
	    if (data & 128)
		pulse_draw_point24( fg );
	    else
		pulse_draw_point24( bg );
	    data <<= 1;
	}
    }
}

void
gfx_putchar(int x, int y, u8 c)
{
    gfx_putcharCD(x, y, c, gfx_fg, gfx_bg);
}

void
gfx_writecharCD(u8 c, color24_t fg, color24_t bg)
{
    if (c == '\n')
    {
	gfx_cursorx = 0;
	gfx_cursory++;
	if (gfx_cursory >= GFX_TEXTHEIGHT)
	    gfx_cursory = 0;
	return;
    }
    if (c == '\b')
    {
	gfx_cursorx--;
	if (gfx_cursorx < 0)
	{
	    gfx_cursorx = GFX_TEXTWIDTH-1;
	    gfx_cursory--;
	    if (gfx_cursory < 0)
		gfx_cursory = 0;
	}
	gfx_putcharCD(gfx_cursorx, gfx_cursory, ' ', fg, bg);
	return;
    }
    gfx_putcharCD(gfx_cursorx, gfx_cursory, c, fg, bg);
    gfx_cursorx++;
    if (gfx_cursorx >= GFX_TEXTWIDTH)
    {
	gfx_cursorx = 0;
	gfx_cursory++;
	if (gfx_cursory >= GFX_TEXTHEIGHT)
	    gfx_cursory = 0;
    }
}

void
gfx_writechar(u8 c)
{
    gfx_writecharCD(c, gfx_fg, gfx_bg);
}

void
gfx_writestringCD(const char *s, color24_t fg, color24_t bg)
{
    while (*s)
    {
	gfx_writecharCD(*s++, fg, bg);
    }
}

void
gfx_writestring(const char *s)
{
    while (*s)
    {
	gfx_writechar(*s++);
    }
}

void
gfx_setcursor(int x, int y)
{
    gfx_cursorx = x;
    gfx_cursory = y;
}

void
gfx_setbg(u8 r, u8 g, u8 b)
{
    gfx_bg.red = r;
    gfx_bg.green = g;
    gfx_bg.blue = b;
}

void
gfx_setfg(u8 r, u8 g, u8 b)
{
    gfx_fg.red = r;
    gfx_fg.green = g;
    gfx_fg.blue = b;
}
//
// Message functions
//
//
// RAM used: 36 bytes
//

#define MSG_MAXLEN	36

u8 msg_data[MSG_MAXLEN];

void
msg_clear()
{
    msg_data[0] = '\0';
}

void
msg_report(const char *s)
{
    int		off;

    for (off = 0; off < MSG_MAXLEN; off++)
    {
	if (!msg_data[off])
	    break;
    }
    // Overflow, return.
    if (msg_data[off])
	return;
    while (off < MSG_MAXLEN)
    {
	msg_data[off++] = *s;
	if (!*s)
	    return;
	s++;
    }
}

void
msg_reportC(const char *s)
{
    if (!*s)
	return;

    if (*s >= 'a' && *s <= 'z')
    {
	char	cap[2];
	cap[0] = *s - 'a' + 'A';
	cap[1] = 0;
	msg_report(cap);
	msg_report(s+1);
    }
    else
	msg_report(s);
}

void
msg_draw()
{
    gfx_setcursor(0, 0);
    int			off;
    for (off = 0; off < MSG_MAXLEN; off++)
    {
	if (!msg_data[off])
	    break;

	gfx_writechar(msg_data[off]);
    }
    for (; off < MSG_MAXLEN; off++)
	gfx_writechar(' ');
}
/**
 * Maze Simulator
 *
 **/

void handle_button_wakeup();

// Must be power of two
#define MAPWIDTH 16
#define MAPHEIGHT 16
// Shift for width!
#define MAPSHIFT 4

u8 glb_mapdata[MAPWIDTH*MAPHEIGHT];

typedef u16	POS;

u8	glb_domaze = 0;
u8	glb_hp = 10;
u8	glb_gold = 0;
u8	glb_depth = 1;

const char *mob_alltypes = "abcdo";

#define POSXY(x, y) ( ((x) + ((y) * MAPWIDTH)) & (MAPWIDTH*MAPHEIGHT-1) )

#define POS2XY(pos, x, y) \
    x = (pos) & (MAPWIDTH-1); \
    y = (pos) >> (MAPSHIFT);

#define MAP(pos) glb_mapdata[pos]
#define MAPXY(x, y) MAP( POSXY(x, y) )

#define FORALL_POS(pos) \
    for ((pos) = 0; (pos) < (MAPWIDTH * MAPHEIGHT); (pos)++)

POS
delta(POS p, int dx, int dy)
{
    p += dx + dy * MAPWIDTH;
    p &= (MAPWIDTH*MAPHEIGHT-1);
    return p;
}

void
status_draw()
{
    char	buf[10];

    gfx_setcursor(0, 15);
    gfx_setfg(255, 0, 0);
    gfx_writestring(SYMBOLSTRING_HEART);
    gfx_setfg(255, 255, 255);
    sprintf(buf, "%d ", glb_hp);
    gfx_writestring(buf);
    gfx_setfg(255, 196, 0);
    gfx_writestring("$");
    gfx_setfg(255, 255, 255);
    sprintf(buf, "%d ", glb_gold);
    gfx_writestring(buf);

    sprintf(buf, SYMBOLSTRING_DLEVEL "%d ", glb_depth);
    gfx_writestring(buf);
}

void
map_clear(u8 tile)
{
    POS		p;
    FORALL_POS(p)
	MAP(p) = tile;
}

bool
map_find(u8 tile, POS *pos)
{
    int		nfound = 0;
    POS		p;

    FORALL_POS(p)
    {
	if (MAP(p) == tile)
	{
	    nfound++;
	    if (!rand_choice(nfound))
	    {
		*pos = p;
	    }
	}
    }
    return nfound != 0;
}

bool
map_findfirst(u8 tile, POS *pos)
{
    POS		p;

    FORALL_POS(p)
    {
	if (MAP(p) == tile)
	{
	    *pos = p;
	    return true;
	}
    }
    return false;
}

void
map_buildcave()
{
    map_clear('#');

    // Seed location
    MAP(0) = '.';
    const int numdrunk = 20;
    const int drunklife = 20;

    for (int drunk = 0; drunk < numdrunk; drunk++)
    {
	POS		pos;
	int		dx, dy;

	map_find('.', &pos);
	for (int life = 0; life < drunklife; life++)
	{
	    rand_dir(&dx, &dy);
	    pos = delta(pos, dx, dy);
	    MAP(pos) = '.';
	}
    }
}

void
map_reset()
{
    glb_hp = 10;
    glb_gold = 0;
    glb_depth = 1;
}

void
map_build()
{
    map_buildcave();

    MAP(0) = '@';

    POS		mpos;

    // Create mobs
    for (int mobtype = 0; mob_alltypes[mobtype]; mobtype++)
    {
	map_find('.', &mpos);
	MAP(mpos) = mob_alltypes[mobtype];
    }

    // Create gold
    for (int ngold = 0; ngold < 5; ngold++)
    {
	map_find('.', &mpos);
	MAP(mpos) = '$';
    }

    // Make downstairs
    map_find('.', &mpos);
    MAP(mpos) = '>';
}

color24_t
tile_colour(u8 val, int x, int y)
{
    color24_t	fg, bg;

    int		offset = gfx_getchartile(val);
    u8		data;

    fg.red = 255;
    fg.green = 255;
    fg.blue = 255;

    bg.red = 0;
    bg.green = 0;
    bg.blue = 0;

    switch (val)
    {
	case '$':
	case 'b':
	    fg.blue = 0;
	    break;
	case 'a':
	    fg.green = 0;
	    fg.blue = 0;
	    break;
	case 'o':
	    fg.red = 0;
	    fg.blue = 0;
	    break;
	case '#':
	    fg.red = 255;
	    fg.blue = 192;
	    fg.blue = 128;
	    break;
    }

    data = alphabet_Tiles[offset * 8 + y];

    if (data & (128 >> x))
    {
	bg = fg;
    }

    bg.alpha = 0;

    return bg;
}

void 
map_draw(POS mpos)
{
    int		mx, my;
    
    POS2XY(mpos, mx, my);

    pulse_set_draw_window(0, 24, 96-1, 24 + 96 - 1);

    for (int y = 0; y < 96; y++)
    {
	int		sy = y - 4 + (my-5)*8; 
	for (int x = 0; x < 96; x++)
	{
	    int		sx = x - 4 + (mx-5)*8; 

	    u8		tile;
	    tile = MAPXY(sx>>3, sy>>3);

	    pulse_draw_point24( tile_colour(tile, sx & 7, sy & 7) );
	}
    }
}

bool
action_walk(POS mpos, int dx, int dy)
{
    POS		goalpos = delta(mpos, dx, dy);
    u8		goal = MAP(goalpos);

    if (goal == '#')
    {
	msg_report("Ouch!  ");
	return false;
    }
    if (goal != '.')
    {
	if (MAP(mpos) == '@')
	{
	    if (goal == '$')
	    {
		msg_report("Gold! ");
		glb_gold++;
	    }
	    else if (goal == '>')
	    {
		msg_report("You descend. ");
		// Generate a new map!
		map_build();
		glb_depth++;
		return false;
	    }
	}
	else
	    return false;
    }

    u8		sym = MAP(mpos);
    MAP(mpos) = '.';
    MAP(goalpos) = sym;

    return true;
}

const char *
mob_getname(u8 mob)
{
    if (mob == '@')
	return "you";
    if (mob == 'o')
	return "orc";
    if (mob == 'a')
	return "ant";
    if (mob == 'b')
	return "bee";
    if (mob == 'c')
	return "cat";
    if (mob == 'd')
	return "dog";
    return "yeti";
}

bool
action_melee(POS mpos, int dx, int dy)
{
    POS		goalpos = delta(mpos, dx, dy);
    u8		target = MAP(goalpos);
    u8		me = MAP(mpos);
    bool	hit = false;

    // Determine if we hit
    if (rand_choice(3))
	hit = true;

    if (hit)
    {
	bool		kills = false;
	if (target != '@')
	{
	    kills = true;
	}
	else
	{
	    glb_hp--;
	    if (glb_hp <= 0)
	    {
		glb_hp = 0;
		kills = true;
	    }
	}

	if (kills)
	{
	    msg_reportC(mob_getname(me));
	    msg_report(" kill ");
	    msg_report(mob_getname(target));
	    msg_report(". ");
	    MAP(goalpos) = '.';
	    // We do not move into the spot!
	    return true;
	}
	else
	{
	    msg_reportC(mob_getname(me));
	    msg_report(" hit ");
	    msg_report(mob_getname(target));
	    msg_report(". ");
	}
    }
    else
    {
	// Just some spam.
	msg_reportC(mob_getname(me));
	msg_report(" miss ");
	msg_report(mob_getname(target));
	msg_report(". ");
    }
    return true;
}

bool
action_bump(POS mpos, int dx, int dy)
{
    POS		goalpos = delta(mpos, dx, dy);
    u8		goal = MAP(goalpos);

    if ((dx || dy) && 
	(goal == '@' || (goal >= 'a' && goal <= 'z') ||
	 (goal >= 'A' && goal <= 'Z')))
    {
	return action_melee(mpos, dx, dy);
    }
    return action_walk(mpos, dx, dy);
}

void
ai_randomwalk(POS mpos)
{
    // Determine adjacent legal square.
    int		dx, dy;
    rand_dir(&dx, &dy);
    POS		goalpos = delta(mpos, dx, dy);
    if (MAP(goalpos) != '#')
    {
	// Walk that way!
	action_bump(mpos, dx, dy);
    }
}

void
map_drawworld()
{
    POS		apos;

    map_findfirst('@', &apos);
    map_draw(apos);

    msg_draw();
    status_draw();
}

void
maze_tick()
{
    // Locate the avatar
    bool	didmove = false;
    u8		c;

    msg_clear();

    c = morse_getchar();

    if (!c)
	return;

    // Ignore the auto space.
    if (c == ' ')
	return;

    int		dx = 0, dy = 0;
    switch (c)
    {
	case 'm':
	    dx = -1;
	    didmove = true;
	    break;
	case 'i':
	    dy = 1;
	    didmove = true;
	    break;
	case 'a':
	    dy = -1;
	    didmove = true;
	    break;
	case 'n':
	    dx = 1;
	    didmove = true;
	    break;

	case '.':
	    didmove = true;
	    break;

	case 'r':
	    map_build();
	    map_reset();
	    break;

	case 'q':
	    glb_domaze = 0;
	    handle_button_wakeup();
	    return;
    }

    POS			mpos;

    if (didmove && map_findfirst('@', &mpos))
    {
	didmove = action_bump(mpos, dx, dy);
    }

    if (didmove)
    {
	for (int mobtype = 0; mob_alltypes[mobtype]; mobtype++)
	{
	    if (map_findfirst(mob_alltypes[mobtype], &mpos))
	    {
		ai_randomwalk(mpos);
	    }
	}
    }

    map_drawworld();
}

void draw_clock()
{
    struct pulse_time_tm current_time;
    pulse_get_time_date(&current_time);

    char		buf[12];

    sprintf(buf, "%d:%02d:%02d\n", current_time.tm_hour, current_time.tm_min, current_time.tm_sec);

    gfx_writestring(buf);

#if 0
    int totalsec = (current_time.tm_hour * 60 + current_time.tm_min) * 60 + current_time.tm_sec;
    int offset = totalsec & 7;
    const char *msg[8] =
    { "Daddy",
      "Sophie",
      "Mommy",
      "Melly",
      "Avi",
      "Amma",
      "Grandma",
      "Grandpa" };

    gfx_writestring(msg[offset]);
    gfx_writechar('\n');
#endif
}

void draw_date()
{
    struct pulse_time_tm current_time;
    pulse_get_time_date(&current_time);

    char		buf[12];

    sprintf(buf, "%d.%d.%d\n", current_time.tm_year+1900, current_time.tm_mon+1, current_time.tm_mday);

    gfx_writestring(buf);
}

void handle_button_wakeup()
{
    // Erase the clock as old crap is left behind.
    pulse_blank_canvas();
    gfx_reset();
    draw_clock();
    gfx_writestring("\n> ");
    morse_reset();
}

// This function is called once after the watch has booted up
// and the OS has loaded
void main_app_init()
{
    pulse_register_callback(ACTION_WOKE_FROM_BUTTON, &handle_button_wakeup);
    rand_seed();
    handle_button_wakeup();
}

u32 glb_downtime = 0;
u32 glb_uptime = 0;

void main_app_handle_button_down()
{
    glb_uptime = 0;
    glb_downtime = pulse_get_millis();

    morse_buttondown(glb_downtime);
}

void main_app_handle_button_up()
{
    glb_uptime = pulse_get_millis();
    morse_buttonup(glb_uptime);
    if (glb_uptime - glb_downtime > 1000)
    {
	glb_domaze = 0;
	pulse_update_power_down_timer(100);
    }
    glb_downtime = 0;
}

bool
command_comp(const char *cmd, const char *comp)
{
    while (*comp && *cmd)
    {
	if (*cmd != *comp)
	    return false;
	comp++;
	cmd++;
    }
    if  (*comp)
	return false;
    return true;
}

void
runcommand(const char *command)
{
    while (*command && *command == ' ')
	command++;

    if (command_comp(command, "rogue"))
    {
	map_build();
	map_reset();
	glb_domaze = true;
	map_drawworld();
    }
    if (command_comp(command, "time"))
    {
	draw_clock();
    }
    if (command_comp(command, "date"))
    {
	draw_date();
    }
    if (command_comp(command, "ls"))
    {
	gfx_writestring("date\nrogue\ntime");
    }
}

char	command[12];
u8	commandpos = 0;

// Main loop. This function is called frequently.
// No blocking calls are allowed in this function or else the watch will reset.
// The inPulse watchdog timer will kick in after 5 seconds if a blocking
// call is made.
void main_app_loop()
{
    static u32	lastms;
    u32		curms;

    curms = pulse_get_millis();

    morse_mainloop(curms);
    if (curms - lastms >= 50)
    {
	if (glb_domaze)
	    maze_tick();
	else
	{
	    u8		c;
	    c = morse_getchar();
	    if (c)
	    {
		gfx_writechar(c);
		if (c == '\b')
		{
		    if (commandpos)
			commandpos--;
		}
		else
		{
		    command[commandpos] = c;
		    commandpos++;
		    if (commandpos >= 12)
			commandpos = 11;
		}

		if (c == '.')
		{
		    gfx_writechar('\n');
		    morse_reset();
		    runcommand(command);
		    gfx_writestring("\n> ");
		    commandpos = 0;
		}
	    }
	}
	lastms = curms;
    }
}

// This function is called whenever the processor is about to sleep (to conserve power)
// The sleep functionality is scheduled with pulse_update_power_down_timer(uint32_t)
void main_app_handle_doz()
{

}

void main_app_handle_hardware_update(enum PulseHardwareEvent event)
{

}
